#ifndef _KEXEC_H
#define _KEXEC_H
#include <mini-os/elf.h>

/*
 * Kexec module used to hand over memory across kexec().
 *
 * This is an ABI which should be modified only in a compatible way.
 * struct kexec_module is located at the start of the last page of the module.
 *
 * The module is usually placed towards the end of memory in order to not
 * conflict with the load address of the new kernel.  In order to allow for a
 * resize by the new kernel after kexec(), the admin data described by struct
 * kexec_module is located at the start of the last page of the module.  This
 * allows to add additional pages towards lower addresses.
 *
 * The kexec module (if configured) is being allocated at boot time in case
 * no kexec module has been passed to the kernel.  If a kexec module has been
 * passed to the kernel, it is kept and will be passed on by kexec().
 *
 * The module can contain two different kinds of information: records and
 * pages.
 *
 * A component wanting to hand over information to the new kernel needs to
 * allocate one or more records.  The layout of a record is defined by the
 * component using it, the layout can be changed only in a compatible way.
 *
 * The kexec module can contain records of multiple components.  Each record
 * type has a unique id which is being used to find the record in the new
 * kernel instance.
 *
 * A record is meant to contain small amounts of information, as all records
 * need to fit into the last page of the kexec module.  In case more data
 * needs to be handed over, one or multiple pages can be allocated within the
 * kexec module.  Module page allocations need to be done with the id of a
 * previously allocated record, which will usually contain some information
 * related to the allocated page(s).
 *
 * All references within the kexec module are coded as offsets relative to the
 * start of the last module page.
 *
 * All admin data (struct kexec_module, record offset table and records) must
 * fit into the last page of the module.
 */
struct kexec_module {
    uint8_t eye_catcher[8];
#define KEXECMOD_EYECATCHER "KexecMem"
    uint16_t n_pages;       /* Number of allocatable pages in the module.    */
    uint16_t n_records;     /* Size of record table (max. 255).              */
#define KEXECMOD_REC_MAX    255
    uint16_t recs_off;      /* Offset to record table from start of page.    */
                            /* The record table is an array of               */
                            /* struct kexec_module_rec.                      */
    uint8_t pg2rec[];       /* Mapping of module pages to associated module  */
                            /* record.  Allocated pages are indicated by     */
                            /* their record number (starting from 0).  Free  */
                            /* pages have value 255.                         */
#define KEXECMOD_PG_FREE    255
};

struct kexec_module_rec {
    uint16_t offset;        /* Offset to record from start of page.          */
    uint8_t type;           /* Type of record.                               */
#define KEXECMOD_REC_NONE   0
#define KEXECMOD_REC_9PFS   1
    uint8_t size;           /* Size of record.                               */
};

extern unsigned long kexec_mod_start;
extern struct kexec_module *mod_ptr;
extern struct kexec_module_rec *mod_recs;
extern char *mod_rec_start;
extern char *mod_rec_end;

/* One element of kexec actions (last element must have action KEXEC_CALL): */
struct kexec_action {
    enum {
        KEXEC_COPY,   /* Copy len bytes from src to dest. */
        KEXEC_ZERO,   /* Zero len bytes at dest. */
        KEXEC_CALL    /* Call dest with paging turned off, param is src. */
    } action;
    unsigned int len;
    void *dest;
    void *src;
};

#ifdef CONFIG_KEXEC
unsigned long kexec_alloc_mod_pages(unsigned int recid, unsigned int n);
void kexec_free_mod_pages(unsigned int recid, unsigned long addr,
                          unsigned int n);
int kexec_find_mod_record(unsigned int start_idx, unsigned int type,
                          unsigned int *size);
int kexec_add_mod_record(unsigned int type, void *addr, unsigned int size);
int kexec_upd_mod_record(unsigned int idx, unsigned int type,
                         void *addr, unsigned int size);
int kexec_read_mod_record(unsigned int idx, void *addr, unsigned int size);

#define KEXEC_MAX_ACTIONS  16

extern char _kexec_start[], _kexec_end[];
extern struct kexec_action kexec_actions[KEXEC_MAX_ACTIONS];
extern unsigned long __kexec_array_start[], __kexec_array_end[];

typedef int(*kexeccall_t)(bool undo);
#define kexec_call(func)                                                   \
    static kexeccall_t __kexeccall_##func __attribute__((__used__))        \
                       __attribute__((__section__(".kexec_array"))) = func

extern unsigned long kexec_last_addr;

int kexec_add_action(int action, void *dest, void *src, unsigned int len);

#define KEXEC_SECSIZE ((unsigned long)_kexec_end - (unsigned long)_kexec_start)

int kexec(void *kernel, unsigned long kernel_size,
          const char *cmdline);

/* Initiate final kexec stage. */
void do_kexec(void *kexec_page);

/* Assembler code for switching off paging and passing execution to new OS. */
void kexec_phys(void);

/* Check kernel to match current architecture. */
bool kexec_chk_arch(elf_ehdr *ehdr);

/* Architecture specific ELF handling functions. */
int kexec_arch_analyze_phdr(elf_ehdr *ehdr, elf_phdr *phdr);
int kexec_arch_analyze_shdr(elf_ehdr *ehdr, elf_shdr *shdr);
bool kexec_arch_need_analyze_shdrs(void);

/* Finalize parameter location and size. */
void kexec_set_param_loc(const char *cmdline);

/* Get entry point and parameter of new kernel. */
int kexec_get_entry(const char *cmdline);
void kexec_get_entry_undo(void);

/* Move used pages away from new kernel area. */
int kexec_move_used_pages(unsigned long boundary, unsigned long kernel,
                          unsigned long kernel_size);
void kexec_move_used_pages_undo(void);

/* Check for kexec module and create kexec memory if needed. */
void kexec_module(unsigned long start_pfn, unsigned long max_pfn);

#else /* CONFIG_KEXEC */
static inline unsigned long kexec_alloc_mod_pages(unsigned int recid,
                                                  unsigned int n)
{
    return 0;
}

static inline void kexec_free_mod_pages(unsigned int recid, unsigned long addr,
                                        unsigned int n)
{
}

static inline int kexec_find_mod_record(unsigned int start_idx,
                                        unsigned int type, unsigned int *size)
{
    return 0;
}

static inline int kexec_add_mod_record(unsigned int type, void *addr,
                                       unsigned int size)
{
    return 0;
}

static inline int kexec_upd_mod_record(unsigned int idx, unsigned int type,
                                       void *addr, unsigned int size)
{
    return 0;
}

static inline int kexec_read_mod_record(unsigned int idx, void *addr,
                                        unsigned int size)
{
    return 0;
}

#endif

#endif /* _KEXEC_H */
